     /*-                                                                            +
      * Copyright (c) 2006 Scott C. Klement                                         +
      * All rights reserved.                                                        +
      *                                                                             +
      * Redistribution and use in source and binary forms, with or without          +
      * modification, are permitted provided that the following conditions          +
      * are met:                                                                    +
      * 1. Redistributions of source code must retain the above copyright           +
      *    notice, this list of conditions and the following disclaimer.            +
      * 2. Redistributions in binary form must reproduce the above copyright        +
      *    notice, this list of conditions and the following disclaimer in the      +
      *    documentation and/or other materials provided with the distribution.     +
      *                                                                             +
      * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND      +
      * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE       +
      * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE  +
      * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE     +
      * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL  +
      * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS     +
      * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)       +
      * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT  +
      * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY   +
      * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF      +
      * SUCH DAMAGE.                                                                +
      *                                                                             +
      */                                                                            +
     H NOMAIN COPYRIGHT('December 29, 2006 by Scott Klement')
     H BNDDIR('QC2LE')

      /COPY SOCKET_H
      /COPY SOCKUTIL_H
      /COPY SIGNAL_H
      /COPY ERRNO_H

     D enable_signals  pr
     D caught_alarm    pr
     D   signo                       10I 0 value

      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Set a File Descriptor in a set ON...  for use w/Select()
      *
      *      fd = descriptor to set on
      *      set = descriptor set
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P FD_SET          B                   EXPORT
     D FD_SET          PI
     D   fd                          10I 0
     D   set                               like(fdset)
     D s               s             10U 0
     D c               s             10U 0 based(p)
      /free
           p = %addr(set) + (%div(fd:32) * %size(c));
           s = 2 ** %rem(fd:32);
           c = %bitor(c:s);
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Set a File Descriptor in a set OFF...  for use w/Select()
      *
      *      fd = descriptor to set off
      *      set = descriptor set
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P FD_CLR          B                   EXPORT
     D FD_CLR          PI
     D   fd                          10I 0
     D   set                               like(fdset)
     D s               s             10U 0
     D c               s             10U 0 based(p)
      /free
           p = %addr(set) + (%div(fd:32) * %size(c));
           s = 2 ** %rem(fd:32);
           c = %bitand(c:%bitnot(s));
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Determine if a file desriptor is on or off...
      *
      *      fd = descriptor to set off
      *      set = descriptor set
      *
      *   Returns *ON if its on, or *OFF if its off.
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P FD_ISSET        B                   EXPORT
     D FD_ISSET        PI             1N
     D   fd                          10I 0
     D   set                               like(fdset)
     D s               s             10U 0
     D c               s             10U 0 based(p)
     D r               s             10U 0
      /free
           p = %addr(set) + (%div(fd:32) * %size(c));
           s = 2 ** %rem(fd:32);
           r = %bitand(c:s);
           return (r = s);
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * Clear All descriptors in a set.  (also initializes at start)
      *
      *      set = descriptor set
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P FD_ZERO         B                   EXPORT
     D FD_ZERO         PI
     D   set                      32767A   options(*varsize)
     D   setsize                     10I 0 value options(*nopass)
     D size            s             10I 0 inz(%size(fdset))
      /free
         if %parms>=2;
            size = setsize;
         endif;
         %subst(set:1:size) = *ALLx'00';
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbsocket():  Create a non-blocking socket
      *
      *    family = (input) communications domain ("protocol family")
      *      type = (input) socket type
      *  protocol = (input) protocol within family
      *
      * Returns new socket descriptor, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P nbsocket        B                   export
     D nbsocket        pi            10i 0
     D    family                     10i 0 value
     D    type                       10i 0 value
     D    protocol                   10i 0 value

     D rc              s             10i 0
     D flags           s             10i 0
      /free

          rc = socket(family: type: protocol);
          if (rc = -1);
             return -1;
          endif;

          flags = fcntl(rc: F_GETFL);
          flags = %bitor(flags: O_NONBLOCK);
          fcntl(rc: F_SETFL: flags);

          return rc;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbconnect(): Connect with a non-blocking socket and timeout
      *
      *    sock = (input) socket to connect
      *    addr = (input) sockaddr structure that denotes the
      *                   location to connect to.
      *    size = (input) size of preceding structure
      *    secs = (input) seconds to wait for connection before
      *                   timing out.
      *
      * Returns 0 if successful, or -1 upon error (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P nbconnect       B                   export
     D nbconnect       pi            10i 0
     D    sock                       10i 0 value
     D    addr                         *   value
     D    size                       10i 0 value
     D    secs                       10i 0 value

     D writeset        s                   like(fdset)
     D timeout         ds                  likeds(timeval)
     D err             s             10i 0 based(p_err)
     D rc              s             10i 0
     D connerr         s             10i 0
     D errsize         s             10i 0

      /free

          p_err = sys_errno();

          // -----------------------------------------------
          // On a non-blocking connection, the connect() API
          // won't wait for completion.  It'll start the
          // connection attempt, then return EINPROGRESS to
          // tell you that the connection is in progress...
          // -----------------------------------------------

          rc = connect(sock: addr: size);
          select;
          when rc = 0;
             return 0;
          when err <> EINPROGRESS;
             return -1;
          endsl;

          // -----------------------------------------------
          //  The select() API can be used to wait for a
          //  connection to complete by waiting until it's
          //  "writable".  Note that the select() API has
          //  a timeout value you can set!
          //
          //  Select returns 0 if a timeout occurs.
          //
          //  Note that select() only returns -1 if an error
          //  occurs with the select() API.  If the connect()
          //  API (running in the background) fails, it will
          //  not return -1.
          // -----------------------------------------------

          FD_ZERO(writeset);
          FD_SET(sock: writeset);
          timeout.tv_sec = secs;
          timeout.tv_usec = 0;

          rc = select( sock+1             // descriptor count
                     : *null              // read set
                     : %addr(writeset)    // write set
                     : *null              // exception set
                     : %addr(timeout) );  // timeout
          select;
          when rc = 0;
             err = ETIME;
             return -1;
          when rc = -1;
             return -1;
          endsl;

          // -----------------------------------------------
          //  To detect if the connect() API (running in the
          //  background) has failed, you need to get the
          //  SO_ERROR socket option
          // -----------------------------------------------

          size = %size(connerr);
          getsockopt( sock
                    : SOL_SOCKET
                    : SO_ERROR
                    : %addr(connerr)
                    : errsize );
          if (connerr <> 0);
             err = connerr;
             return -1;
          endif;

          return 0;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbaccept(): Wait for a new client connection or timeout
      *             on a non-blocking socket
      *
      *    sock = (input) listener socket to accept from
      *    addr = (output) sockaddr structure that specifies who
      *                    the connection is from (or *NULL)
      *    size = (in/out) size of preceding structure, or *OMIT
      *                    if preceding structure is *NULL
      *    secs = (input) seconds to wait for new connection
      *                   before timing out.
      *
      * Returns new socket descriptor if successful,
      *         or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P nbaccept        B                   export
     D nbaccept        pi            10i 0
     D    sock                       10i 0 value
     D    addr                         *   value
     D    size                       10i 0 options(*omit)
     D    secs                       10i 0 value

     D readset         s                   like(fdset)
     D timeout         ds                  likeds(timeval)
     D err             s             10i 0 based(p_err)
     D rc              s             10i 0
     D cli             s             10i 0

      /free

          p_err = sys_errno();

          dow '1';

              cli = accept(sock: addr: size);
              if (cli <> -1);
                  return cli;
              endif;

              if (err <> EWOULDBLOCK);
                  return -1;
              endif;

              FD_ZERO(readset);
              FD_SET(sock: readset);
              timeout.tv_sec = secs;
              timeout.tv_usec = 0;

              rc = select( sock+1             // size of fd set
                         : %addr(readset)     // read set
                         : *null              // write set
                         : *null              // exception set
                         : %addr(timeout) );  // timeout
              select;
              when rc = 0;
                 err = ETIME;
                 return -1;
              when rc = -1;
                 return -1;
              endsl;

          enddo;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbrecv(): Receive data on non-blocking socket w/timeout
      *
      *    sock = (input) socket to receive data from
      *     buf = (output) buffer to receive into (address of variable)
      *    size = (input) size of buffer to send (size of variable)
      *    secs = (input) seconds to wait before timing out
      *
      * Returns length received, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P nbrecv          B                   export
     D nbrecv          pi            10i 0
     D    sock                       10i 0 value
     D    buf                          *   value
     D    size                       10i 0 value
     D    secs                       10p 3 value

     D readset         s                   like(fdset)
     D timeout         ds                  likeds(timeval)
     D err             s             10i 0 based(p_err)
     D rc              s             10i 0

      /free

          p_err = sys_errno();

          dow '1';

              rc = recv(sock: buf: size: 0);
              if (rc <> -1);
                 return rc;
              endif;
              if (err <> EWOULDBLOCK);
                 return rc;
              endif;

              // -----------------------------------
              //  Wait until socket is readable
              // -----------------------------------

              FD_ZERO(readset);
              FD_SET(sock: readset);
              timeout.tv_sec = %int(secs);
              timeout.tv_usec = (secs - timeout.tv_sec) * 1000000;

              rc = select( sock+1             // descriptor count
                         : %addr(readset)     // read set
                         : *null              // write set
                         : *null              // exception set
                         : %addr(timeout) );  // timeout
              select;
              when rc = 0;
                 err = ETIME;
                 return -1;
              when rc = -1;
                 return -1;
              endsl;

          enddo;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbsend(): Send data on non-blocking socket w/timeout
      *
      *    sock = (input) socket to send data on
      *     buf = (input) buffer to send
      *    size = (input) size of buffer to send
      *    secs = (input) seconds to wait before timing out
      *
      * Returns length sent, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P nbsend          B                   export
     D nbsend          pi            10i 0
     D    sock                       10i 0 value
     D    buf                          *   value
     D    size                       10i 0 value
     D    secs                       10p 3 value

     D writeset        s                   like(fdset)
     D timeout         ds                  likeds(timeval)
     D err             s             10i 0 based(p_err)
     D rc              s             10i 0
     D sent            s             10i 0

      /free

          p_err = sys_errno();

          dow '1';

              // -----------------------------------
              //  Send data.
              //   If all data sent, great, return.
              //   If partial data sent, adjust buffer
              // -----------------------------------

              rc = send(sock: buf: size: 0);
              if (rc=-1 and err<>EWOULDBLOCK);
                 return -1;
              endif;

              if (rc <> -1);
                  sent = sent + rc;
                  if (rc = size);
                      return sent;
                  endif;
                  size = size - rc;
                  buf  = buf  + rc;
              endif;

              // -----------------------------------
              //  Wait until socket is writable again
              //  (with timeout)
              // -----------------------------------

              FD_ZERO(writeset);
              FD_SET(sock: writeset);
              timeout.tv_sec = %int(secs);
              timeout.tv_usec = (secs - timeout.tv_sec) * 1000000;

              rc = select( sock+1             // descriptor count
                         : *null              // read set
                         : %addr(writeset)    // write set
                         : *null              // exception set
                         : %addr(timeout) );  // timeout
              select;
              when rc = 0;
                 err = ETIME;
                 return -1;
              when rc = -1;
                 return -1;
              endsl;

          enddo;

      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * enable_signals(): This is called internally when we want
      *                   to enable signal processing.
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P enable_signals  B
     D enable_signals  pi
     D act             ds                  likeds(sigaction_t)
     D enabled         s              1n   inz(*Off) static
      /free
         if (enabled);
            return;
         endif;

         act.sa_flags = 0;
         act.sa_sigaction = *NULL;
         act.sa_handler = %paddr(caught_alarm);
         sigfillset(act.sa_mask);
         sigaction(SIGALRM: act: *omit);

         enabled = *on;
      /end-free
     p                 e


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * This is called when an alarm signal (SIGALRM) is received
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P caught_alarm    B
     D caught_alarm    PI
     D   signo                       10I 0 value
      /free
         // do nothing -- we ignore alarm signals.
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * bsocket():  Create a blocking socket
      *
      *    family = (input) communications domain ("protocol family")
      *      type = (input) socket type
      *  protocol = (input) protocol within family
      *
      * Returns new socket descriptor, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P bsocket         B                   export
     D bsocket         pi            10i 0
     D    family                     10i 0 value
     D    type                       10i 0 value
     D    protocol                   10i 0 value

     D rc              s             10i 0
     D flags           s             10i 0
      /free

          rc = socket(family: type: protocol);
          if (rc = -1);
             return -1;
          endif;

          flags = fcntl(rc: F_GETFL);
          flags = %bitand(flags: %bitnot(O_NONBLOCK));
          fcntl(rc: F_SETFL: flags);

          return rc;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * bconnect(): Connect with a blocking socket w/timeout
      *
      *    sock = (input) socket to connect
      *    addr = (input) sockaddr structure that denotes the
      *                   location to connect to.
      *    size = (input) size of preceding structure
      *    secs = (input) seconds to wait for connection before
      *                   timing out.
      *
      * Returns 0 if successful, or -1 upon error (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P bconnect        B                   export
     D bconnect        pi            10i 0
     D    sock                       10i 0 value
     D    addr                         *   value
     D    size                       10i 0 value
     D    secs                       10i 0 value

     D err             s             10i 0 based(p_err)
     D saveerr         s                   like(err)
     D rc              s             10i 0

      /free
          enable_signals();
          p_err = sys_errno();

          alarm(secs);
          rc = connect(sock: addr: size);
          saveerr = err;
          alarm(0);

          if (rc = -1 and saveerr=EINTR);
             err = ETIME;
          else;
             err = saveerr;
          endif;

          return rc;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * baccept(): Accept connection with a blocking socket w/timeout
      *
      *    sock = (input) listener socket to accept from
      *    addr = (output) sockaddr structure that specifies who
      *                    the connection is from (or *NULL)
      *    size = (in/out) size of preceding structure, or *OMIT
      *                    if preceding structure is *NULL
      *    secs = (input) seconds to wait for new connection
      *                   before timing out.
      *
      * Returns new socket descriptor if successful,
      *         or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P baccept         B                   export
     D baccept         pi            10i 0
     D    sock                       10i 0 value
     D    addr                         *   value
     D    size                       10i 0 options(*omit)
     D    secs                       10i 0 value

     D err             s             10i 0 based(p_err)
     D saveerr         s                   like(err)
     D rc              s             10i 0

      /free
          enable_signals();
          p_err = sys_errno();

          alarm(secs);
          rc = accept(sock: addr: size);
          saveerr = err;
          alarm(0);

          if (rc = -1 and saveerr=EINTR);
             err = ETIME;
          else;
             err = saveerr;
          endif;

          return rc;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbrecv(): Receive data on non-blocking socket w/timeout
      *
      *    sock = (input) socket to receive data from
      *     buf = (output) buffer to receive into (address of variable)
      *    size = (input) size of buffer to send (size of variable)
      *    secs = (input) seconds to wait before timing out
      *
      * Returns length received, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P brecv           B                   export
     D brecv           pi            10i 0
     D    sock                       10i 0 value
     D    buf                          *   value
     D    size                       10i 0 value
     D    secs                       10p 3 value

     D err             s             10i 0 based(p_err)
     D saveerr         s                   like(err)
     D rc              s             10i 0

      /free
          enable_signals();
          p_err = sys_errno();

          alarm(secs);
          rc = recv(sock: buf: size: 0);
          saveerr = err;
          alarm(0);

          if (rc = -1 and saveerr=EINTR);
             err = ETIME;
          else;
             err = saveerr;
          endif;

          return rc;
      /end-free
     P                 E


      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      * nbsend(): Send data on non-blocking socket w/timeout
      *
      *    sock = (input) socket to send data on
      *     buf = (input) buffer to send
      *    size = (input) size of buffer to send
      *    secs = (input) seconds to wait before timing out
      *
      * Returns length sent, or -1 upon failure (check errno!)
      *+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     P bsend           B                   export
     D bsend           pi            10i 0
     D    sock                       10i 0 value
     D    buf                          *   value
     D    size                       10i 0 value
     D    secs                       10p 3 value

     D err             s             10i 0 based(p_err)
     D saveerr         s                   like(err)
     D rc              s             10i 0

      /free
          enable_signals();
          p_err = sys_errno();

          alarm(secs);
          rc = send(sock: buf: size: 0);
          saveerr = err;
          alarm(0);

          if (rc = -1 and saveerr=EINTR);
             err = ETIME;
          else;
             err = saveerr;
          endif;

          return rc;
      /end-free
     P                 E
