/**
 *   code.js = Code for Sample PhoneGap App
 *   
 *                       Scott Klement, May 2017
 *                   
 */


/**
 * beginSetup(): This is called when JavaScript is loaded.  It 
 *               waits until the device is ready, then loads the
 *               product list.
 */

function beginSetup() {
  
  document.addEventListener( "deviceready", 
    function() {
      StatusBar.overlaysWebView(false);
      loadList();
    },
    false
  );
}


/**
 * popupMsg(): Utility to show a message in a popup box.
 *             If run from a device, it will use the Dialog plugin
 *             plugin to get a native device confirmation box.
 *             
 *             If the plugin is not there, it will use the browser's alert()
 *             
 * @param {String} msg = message to show in box.             
 */

function popupMsg(msg) {
  
  if (navigator && navigator.notification) {
    navigator.notification.alert( 
        msg,
        function(results) { /* runs after OK is hit */},
        ""
    );
  }
  else {
    alert(msg);
  }
  
}


/**
 * ajaxCall(): Makes an AJAX (HTTP GET/POST) call to the IBM i
 * 
 * @param {String} method = GET, POST, PUT, DELETE, etc 
 * @param {String} url = URL to connect to
 * @param {String} data = for a POST request, data to send 
 * @param {function} handler = function called with results
 * 
 */

function ajaxCall(method, url, data, handler) {

  var ajax = new XMLHttpRequest();
  ajax.open(method, url, true);
  ajax.setRequestHeader("content-type", "text/plain");

  ajax.onreadystatechange = function() {
    if (ajax.readyState == 4) {
       if (ajax.status == 200) {
         handler(ajax.responseText);
       }
       else if (ajax.status == 0) {
         popupMsg( "HTTP Error:\n"
                 + "Unable to reach server!"
                 );
       }
       else {
         popupMsg( "HTTP Error:\n"
                 + "Status: " + ajax.status + "\n"
                 + "Text:" + ajax.statusText 
                 );
      }
    }
  }

  if (data) {
    ajax.send(data);
  }
  else {
    ajax.send();
  }
  
}


/**
 * appInfo():  Display information about this app
 */

function appInfo() {
  popupMsg( "Version 1.0\n" 
          + "Copyright (c) 2017 - Scott Klement\n"
          + "\n"
          + "This app is for demonstration purposes only. There is "
          + "no technical support offered. Please use it to learn "
          + "so that you can write your own apps."
          );
}


/**
 * closeItem(): Called when the X is clicked to close
 *              an item that's being viewed
 */

function closeItem() {
  
  /* if on the item list screen, remove the list
   * from storage.
   */
  
  if (document.getElementById("itemList")) {
    try {
      localStorage.removeItem("product-list");    
    }
    catch (err) {
      /* ignore */
    } 
  }
  
  loadList();
}


/**
 * scanItem(): Invoke the barcodeScanner plugin to scan a barcode
 * 
 * NOTE: Be sure you add the scanner to your project with
 *       phonegap plugin add cordova-plugin-barcodescanner
 */

function scanItem() {
  
  /* Make sure the plugin exists */
  var scanner = null;
  if (cordova && cordova.plugins && cordova.plugins.barcodeScanner) {
    scanner = cordova.plugins.barcodeScanner;  
  }
  if (!scanner) {
    // scanGood({ cancelled: 0, text: "0300450002539" });
    popupMsg("Barcode scanner plugin not found!");
    return;
  }
  
  
  /* There's no error -- use the scan to load a new 
   * item (unless cancelled) */
   
  function scanGood(result) {
  
    if (result.cancelled) {
      loadList();
      return;
    }
    
    // we only use the last 5 digits
    // of the UPC code (without check digit)
    
    var code = result.text;
    if (code.length > 5) {
      code = code.substr(7, 5);
    }
    
    viewItem(code);
    
  }
  
  /* There was an error -- display the error */
  
  function scanError(msg) {
    popupMsg("Scan failed: " + msg);
  }
  
  /* do the scan  */
  
  scanner.scan( scanGood, scanError);
  
}


/**
 * loadList(): Load the list of items stored on this device
 * 
 * This displays a list of items on the device and lets you click one 
 * to view item information.  The list of items is not loaded from 
 * the server at this time -- but instead the last list that had been
 * loaded will be used.
 * 
 */

function loadList() {

   var jsonData = null;
   try {
     jsonData = localStorage.getItem("product-list");
   }
   catch(err) {
     jsonData = null;
   }

   var table = "<table id=\"itemList\">";

   if (!jsonData) {
      table += "<tr><td>(no data found)</td></tr>";
   }

   if (jsonData) {
      var list = JSON.parse(jsonData);
      for (var i=0; i<list.item.length; i++) {
        table += "<tr>"
               +   "<td class=\"listing\" " 
               +       "onclick=\"viewItem('" + list.item[i].code + "')\">" 
               +      list.item[i].name 
               +   "</td>"
               + "</tr>";
      }
   }

   table += "</table>";

   var middleDiv = document.getElementById("middle")
   if (middleDiv) middleDiv.innerHTML = table;

}


/**
 * refreshList():  Download a new list of items from the server
 * 
 * This routine
 *    1) Calls an RPG web service to get a list of items
 *    2) Saves the list of items in the device's local storage
 *    3) Runs the loadList() routine to display them
 */

function refreshList() {
  
  ajaxCall("GET", "http://power8:8500/webservices/prodinfo/json/list", null, function(data) {

    try {
      var list = JSON.parse(data);
    }
    catch(err) {
      popUp("JSON parse: " + err);
      return;
    }
    
    if (!list.success) {
      popupMsg("Error: " + list.errMsg);
      return;
    }

    localStorage.setItem("product-list", data);
    loadList();
  });
}


/**
 * viewItem(): View a single item
 * 
 * @param {String} item = item to view
 * 
 * This routine:
 *   1. Finds the item in the local storage
 *   2. Shows the item details on the display
 * 
 */

function viewItem(item) {

  /* This will strip any leading zeroes, etc */
  
  if (!isNaN(item)) {
    item = String(parseInt(item));
  }

  
  /* Load the item list from local storage */
  
  var jsonData = null;
  try {
    jsonData = localStorage.getItem("product-list");
  }
  catch(err) {
    popupMsg("No products have been loaded.");
    return;
  }
  
  var list = JSON.parse(jsonData);
  
  
  /* search the item list for the item */
  
  var prod = null;
  for (var i=0; i<list.item.length; i++) {
    if (list.item[i].code == item) {
      prod = list.item[i];
    }
  }
  
  if (!prod) {
    popupMsg("Product " + item + " not found!");
    return;
  }
  
  
  /* load the item onto the display */
  
  function addRow(label, field) {
    var row = "<tr>"
            +   "<td>"
            +     "<span class=\"label\">" + label + "</span>"
            +     "<span class=\"field\">" + field + "</span>"
            +   "</td>"
            + "</tr>";
    return row;
  }

  var middleDiv = document.getElementById("middle");
  if (middleDiv) {
     var table = "<table id=\"itemView\">"
               + addRow("PRODUCT CODE", prod.code)
               + addRow("PRODUCT NAME", prod.name)
               + addRow("RETAIL PRICE", prod.price)
               + addRow("STOCK QUANTITY", prod.stock)
               + addRow("IMAGE ID", prod.image)
               + "</table>";
     middleDiv.innerHTML = table;
  }

}
  


/**
 * SERVERBASED_viewItem(): View a single item
 * 
 * @param {String} item = item to view
 * 
 * This routine:
 *   1. Calls an RPG web service to get an item
 *   2. Shows the item details on the display
 * 
 * NOTE: This is a copy of viewItem() that gets the
 *       item directly from the server. It is not 
 *       currently used, but if you replace viewItem
 *       with this one, you can try it out.
 */
  
function SERVERBASED_viewItem(item) {
  
  if (!isNaN(item)) item = String(parseInt(item));
  
  var url = "http://power8:8500/webservices/prodinfo/json/list/" + item;
  
  ajaxCall("GET", url, null, function(data) {
    
    try {
      var result = JSON.parse(data);
    }
    catch (err) {
      popupMsg("JSON parse: " + err);
      return;
    }
    
    if (!result.success) {
      popupMsg("Error: " + result.errMsg);
      return;
    }
    
    var prod = result.item[0];
    
    function addRow(label, field) {
      var row = "<tr>"
              +   "<td>"
              +     "<span class=\"label\">" + label + "</span>"
              +     "<span class=\"field\">" + field + "</span>"
              +   "</td>"
              + "</tr>";
      return row;
    }

    var middleDiv = document.getElementById("middle");
    if (middleDiv) {
       var table = "<table id=\"itemView\">"
                 + addRow("PRODUCT CODE", prod.code)
                 + addRow("PRODUCT NAME", prod.name)
                 + addRow("RETAIL PRICE", prod.price)
                 + addRow("STOCK QUANTITY", prod.stock)
                 + addRow("IMAGE ID", prod.image)
	               + "</table>";
       middleDiv.innerHTML = table;
    }

  });
}

